        ; HP48 - I2C interface PIC code
        ; Processor 16C84
        ; Clock 4MHz (1us instruction cycle)
        ; Copyright (C) 1996 A.R.Duell

        list c=80, n=60, p=16c84, r=hex
        
        ; destination mode bit 
w       equ 0
f       equ 1

        ; Register Definitions
ind_dat equ 0
pc      equ 2
status  equ 3
ind_add equ 4
porta   equ 5
portb   equ 6
trisa   equ 5
trisb   equ 6
cmd     equ 0c ; Current command character
temp    equ 10         
bit_cnt equ 11 ; Serial bit counter
dly_cnt equ 12 ; Delay counter
temp1   equ 13
err_reg equ 14 ; Error code
this_byte equ 15 ; Current data item buffer
last_byte equ 16 ; last+1 address used in data buffer
first_dat equ 20 ; first address used for the data values.
sec_dat   equ 21 ; second data byte (= i2c address on read)

        ; Next routine state values
        ; Each main routine (called by dispatch) 
        ; exits with one of these values in the W
        ; register to indicate what routine to call
        ; next. 
st_cmd  equ 0 ; read command state
st_data equ 1 ; read data state
st_dec  equ 2 ; decode command (and execute) state
st_eom  equ 3 ; wait for end-of-message (<CR>) state
st_err  equ 4 ; send error report to host state
st_ok   equ 5 ; send OK to host state 

        ; Status Register Bits
carry   equ 0 ; carry flag 
zero    equ 2 ; Zero flag
rp      equ 5 ; Data register page select

        ; Port Bit Definitions
TxD     equ 0 ; Port A bit 0 = RS232 output
RxD     equ 1 ; Port A bit 1 = RS232 input
SDA     equ 0 ; Port B bit 0 = I2C data line
SCL     equ 1 ; Port B bit 1 = I2C clock line

        ; Delay Counts
txdlycnt equ 1E ; TX bit delay
rxdlyhlf equ 10 ; RX half bit delay
rxdlycnt equ 1E ; RX bit delay
i2ccnt equ 06   ; I2C delay count

        ; Error Codes
ill_asc equ 01 ; Illegal Ascii Character in hex number
odd_len equ 02 ; Odd length hex string
too_lon equ 03 ; Too many data bytes
ill_cmd equ 04 ; illegal command
ill_len equ 05 ; Incorrect number of parameters to read commands
i2c_err equ 06 ; No acknowledge on I2C bus

Reset   org 0 ; Reset enters here 
        bsf porta,TxD ; RS232 output should be 1 in the idle state
        bsf status,rp ; Select tristate registers       
        movlw B'11111110' ; Set TxD as an output
        movwf trisa
        bcf status,rp ; Select ports
        clrf portb ; clear I2C port lines, so that toggling the
                   ; tristate register bits will cause them to 
                   ; act as O/C outputs.        

        clrw       ; The program should start in the st_cmd state
main_lp call dispatch ; Go to the right routine for the current state
        goto main_lp ; and round again

        ; ****************************
        ; dispatch - go to the next
        ; routine. Each routine must 
        ; exit with W containing the
        ; state number of the next
        ; one. 
        ; This routine must be entirely
        ; in page 0
        ; ****************************
dispatch addwf pc,F
        goto rd_cmd   ; read command value
        goto rd_data  ; read command parameters
        goto dec_exec ; decode and execute command
        goto wait_eom ; wait for end of message after an error
        goto send_err ; send error report to host
        goto send_ok  ; send OK to host

        ; ****************************
        ; nib2asc - convert a nybble 
        ; in W to ASCII in W
        ; This routine must be entirely
        ; in page 0
        ; ****************************
nib2asc addwf pc,F
        retlw 30 ; Ascii 0
        retlw 31
        retlw 32
        retlw 33
        retlw 34
        retlw 35
        retlw 36
        retlw 37
        retlw 38
        retlw 39 ; Ascii 9
        retlw 41 ; Ascii A
        retlw 42
        retlw 43
        retlw 44
        retlw 45
        retlw 46 ; Ascii F

        ; ****************************
        ; asc2nib - convert ascii
        ; character into a nybble
        ; Enter with character in W
        ; Exit with nybble in W
        ; ****************************
asc2nib addlw 0D0 ; Effectivly subtract 30 (hex). Carry will be set
                  ; if the original character was >'0'
        btfss status,carry ; See if the character is too low
        goto a2n_err       ; Exit if error
        movwf temp         ; Save character
        addlw 0F6          ; Subtract 10
        movf temp,W        ; restore character
        btfss status,carry ; Skip if >9
        return             ; It was 0-9, so exit
        addlw 0EF          ; Now see if it's A-F
        btfss status,carry ; No, it's too low
        goto a2n_err       ; Exit if out of range
        movwf temp
        addlw 0FA          ; Test if it's > F
        movf temp,W
        btfsc status,carry ; if it's out of range, give error 
        goto a2n_err
        addlw 0A           ; if it's 0-5, then add 10 to make it A-F
        return             ; and exit        
a2n_err movlw ill_asc      ; Set Error code
        movwf err_reg
        return             ; and exit

        ; ****************************
        ; sendbyte - send 1 byte (in W)
        ; to the serial port in hex
        ; ****************************
sendbyte movwf temp1 ; store byte to send
         swapf temp1,W ; Get MS nybble into low bits
         call nextnib ; send it
         movf temp1,W ; Restore original byte 
                      ; fall-thru into nextnib and send low nybble
nextnib  andlw 0F ; convert to a nybble
         call nib2asc ; Now it's an ascii character
         goto send_rs ; send it

        ; ****************************
        ; rd_cmd - read a command from 
        ; the rs232 port and store it in
        ; the cmd register. Send OK if
        ; a carriage return is found
        ; ****************************
rd_cmd  movlw first_dat ; set up indirect address pointer for the 
        movwf ind_add   ; data routines
        clrf  err_reg   ; clear error flag
        call rec_rs     ; get next character from the serial port
        xorlw 0D        ; is is a carriage return?
        btfsc status,zero ; if so
        retlw st_ok     ; Exit and send OK to the host
        xorlw 0D        ; restore original character
        movwf cmd       ; store it in the command register
        retlw st_data   ; exit to the read data state

        ; ****************************
        ; rd_data - read data parameters 
        ; in hex from the rs232 port
        ; and store them in the data 
        ; buffer area.
        ; , and <sp> are ignored as 
        ; first characters, illegal
        ; as second ones
        ; Will exit into state
        ; st_dat if valid data and not CR
        ; st_dec if first character was a CR
        ; st_err if second character was a CR
        ; st_eom if other error
        ; ****************************
rd_data clrf this_byte ; clear tempoary buffer
        call rec_rs   ; get first character from the serial port
        xorlw 0D      ; is it a carriage return
        btfsc status,zero ; if so
        retlw st_dec  ; exit in the st_dec state to decode command
        xorlw 02D     ; now test if it's a space
        btfsc status,zero ; if so
        retlw st_data ; exit back to st_data state to skip that char
        xorlw 0C      ; now test if it's a comma
        btfsc status,zero ; if so
        retlw st_data ; and skip it if so
        xorlw 02C     ; restore original character
        call asc2nib  ; convert it to hex
        movwf this_byte ; Save it in a temporary store
        swapf this_byte,F ; into the high nybble
        call rec_rs   ; get next character from the serial port
        xorlw 0D      ; test if it's a carriage return
        btfsc status,zero ; if so
        goto len_err  ; exit with an error condition
        xorlw 0D      ; otherwise restore original character
        call asc2nib  ; convert it to hex
        iorwf this_byte,F ; and store it in the buffer
        movf err_reg,F ; test the error flag
        btfss status,zero ; if it's not OK
        retlw st_eom  ; wait for end-of-message and exit
        btfsc ind_add,4 ; test for overflow of the buffer address
        goto too_many ; give error if so
        movf this_byte,W ; otherwise get the character
        movwf ind_dat ; store it in the buffer
        incf ind_add,F ; increment the pointer
        retlw st_data ; and round again for more data
len_err movlw odd_len ; set error flag
        movwf err_reg
        retlw st_err  ; and exit to the error routine
too_many movlw too_lon ; set error flag
        movwf err_reg
        retlw st_eom ; and wait for the end of the message

        ; ****************************
        ; wait_eom - wait for end-of-message
        ; (Carriage Return) and then go to 
        ; the error report state
        ; ****************************
wait_eom call rec_rs  ; get next byte from the serial port
        xorlw 0D      ; is it a carriage return?
        btfss status,zero ; If not...
        goto wait_eom ; round again for another one
        retlw st_err  ; When we've got a carriage return, exit in the
                      ; st_err state

        ;*****************************
        ; dec_exec - decode command 
        ; byte, goto appropriate
        ; routine
        ; The following commands are
        ; implemented
        ; W - start, write data, stop
        ; S - start, write data
        ; P - write data, stop
        ; T - write data
        ; R - start, send address, read data, stop
        ; B - start, send address, read data
        ; E - read data, stop
        ; F - read data
        ; ****************************
dec_exec movf cmd,W   ; get command code
        xorlw 57      ; is it W ?
        btfsc status,zero ; if so
        goto w_cmd    ; execute W command
        xorlw 04      ; is it S?
        btfsc status,zero ; if so
        goto s_cmd    ; execute S command
        xorlw 03      ;  is it P?
        btfsc status,zero ; if so
        goto p_cmd    ; execute P command
        xorlw 04      ; is it T?
        btfsc status,zero ; if so
        goto t_cmd    ; execute T command
        xorlw 06      ; is it R ?
        btfsc status,zero ; if so
        goto r_cmd    ; execute R command
        xorlw 10      ; is it B ? 
        btfsc status,zero ; if so
        goto b_cmd    ; execute B command
        xorlw 07      ; is it E ? 
        btfsc status,zero ; if so
        goto e_cmd    ; execute E command
        xorlw 03      ; is it F ?
        btfsc status,zero ; if so
        goto f_cmd    ; execute F command
        movlw ill_cmd ; set up error register
        movwf err_reg
        retlw st_err  ; and send error to host

        ; ****************************
        ; p_cmd - Perform P command
        ; ****************************
p_cmd   movf ind_add,W  ; get last address + 1 used
        movwf last_byte ; save it
        addlw 0DF       ; check to make sure there is at least 1 data byte
        btfss status,carry ; if not
        goto param_err  ; goto error routine
        goto w_blk      ; if OK, send block
  
        ; ****************************
        ; w_cmd - Perform W command
        ; ****************************
w_cmd   movf ind_add,W  ; get last address + 1 used
        movwf last_byte ; save it
        addlw 0DE       ; check to make sure there are at least 2 data bytes
        btfss status,carry ; if not
        goto param_err  ; goto error routine
        call i2c_st     ; send I2C bus start
w_blk   movlw first_dat ; set up indirect pointer
        movwf ind_add
w_loop  movf ind_dat,W  ; get next byte to send
        call send_i2c   ; send it
        movwf temp1     ; save return code
        incf ind_add,F  ; increment the pointer
        movf ind_add,W  ; get new value
        xorwf last_byte,W ; test to see if all the bytes have been sent
        btfsc status,zero ; if so
        goto w_end      ; finish 
        movf temp1,F    ; otherwise test the return code from send_i2c
        btfss status,zero ; if it acknowledged
        goto w_loop     ; round again
        goto i2c_nack   ; otherwise give error
w_end   call i2c_sp     ; send bus stop condition
        retlw st_ok     ; and exit in the OK state.

        ; ****************************
        ; t_cmd - Perform T command
        ; ****************************
t_cmd   movf  ind_add,W ; get last address + 1 used
        movwf last_byte ; save it
        addlw 0DF       ; check to make sure there is at least 1 data byte
        btfss status,carry ; if not
        goto param_err  ; goto error routine
        goto s_blk      ; if OK, send block

        ; ****************************
        ; s_cmd - Perform S command
        ; ****************************
s_cmd   movf ind_add,W  ; get last used address + 1
        movwf last_byte ; save it
        addlw 0DE       ; check to make sure there are at least 2 data bytes
        btfss status,carry ; if not
        goto param_err  ; goto error routine
        call i2c_st     ; send I2C bus start
s_blk   movlw first_dat ; set up indirect pointer
        movwf ind_add
s_loop  movf ind_dat,W  ; get next byte to send
        call send_i2c   ; send it
        andlw 0FF       ; test return code
        btfsc status,zero ; if it didn't acknowledge
        goto i2c_nack   ; handle the error
        incf ind_add,F  ; increment the pointer
        movf ind_add,W  ; get the new value
        xorwf last_byte,W ; have all the bytes been sent? 
        btfss status,zero ; if not
        goto s_loop     ; round again
        retlw st_ok     ; if so, exit in OK state

        ; ****************************
        ; r_cmd - Perform the R 
        ; command
        ; This command has 2 parameters
        ; - the number of bytes to receive
        ; and the address to receive them from
        ; ****************************
r_cmd   movf ind_add,W  ; get last byte used +1
        xorlw 022       ; check for exactly 2 parameters
        btfss status,zero ; if not
        goto param_err  ; give error
        movf first_dat,F ; make sure that some data has to be received
        btfsc status,zero ; if not 
        goto param_err  ; give error
        call i2c_st     ; send i2c start condition
        movf sec_dat,W  ; get the slave address
        call send_i2c   ; and send it
        andlw 0FF       ; test the return code
        btfsc status,zero ; if it didn't acknowledge
        goto i2c_nack   ; give error
r_loop  movf first_dat,W ; see if this 
        xorlw 01        ; is the last byte
        movlw 01        ; Acknowledge by default
        btfsc status,zero ; if it is the last byte
        movlw 00        ; do not acknowledge
        call rec_i2c    ; get a byte from the i2c bus
        call sendbyte   ; and send it to the host
        decfsz first_dat,F ; decrement byte counter
        goto r_loop     ; and round again
        call i2c_sp     ; send i2c stop condition
        goto send_cr    ; send carriage return, exit in command state

        ; ****************************
        ; e_cmd - Perform E command
        ; This command has 1 parameter
        ; - the number of bytes to receive
        ; ****************************
e_cmd   movf ind_add,W ; look at last byte used
        xorlw 021      ; is there exactly 1 data byte?
        btfss status,zero ; if not
        goto param_err ; give error
        movf first_dat,F ; make sure that some data has to be received
        btfsc status,zero ; if not 
        goto param_err  ; give error
        goto r_loop

        ; ****************************
        ; b_cmd - perform B command
        ; This command has 2 parameters
        ; (like the W command)
        ; ****************************
b_cmd   movf ind_add,W ; look at the last byte used
        xorlw 022      ; check there are exactly 2 data bytes
        btfss status,zero ; if not
        goto param_err ; give an error
        movf first_dat,F ; check that #bytes is not zero
        btfsc status,zero ; if so
        goto param_err ; give an error
        call i2c_st    ; send i2c start condition
        movf sec_dat,W ; get the slave address
        call send_i2c  ; and send it on the bus
        andlw 0FF      ; test the return code
        btfsc status,zero ; if no acknowledge
        goto i2c_nack  ; give an error
b_loop  movlw 01       ; always acknowledge
        call rec_i2c   ; get a byte from the slave
        call sendbyte  ; and send it to the host
        decfsz first_dat,F ; decrement the byte counter
        goto b_loop    ; and round again
        goto send_cr   ; send carriage return and exit in the cmd state

        ; ****************************
        ; f_cmd - Perform F command
        ; 1 parameter - the number of 
        ; bytes to receive
        ; ****************************
f_cmd   movf ind_add,W ; look at last byte used
        xorlw 021      ; is there exactly 1 data byte?
        btfss status,zero ; if not
        goto param_err ; give error
        movf first_dat,F ; make sure that some data has to be received
        btfsc status,zero ; if not 
        goto param_err  ; give error
        goto b_loop
   
        ; ****************************
        ; i2c_nack - handle I2C bus errors
        ; ****************************
i2c_nack call i2c_sp    ; send bus stop 
        movlw i2c_err   ; set up error flag
        movwf err_reg
        retlw st_err    ; and exit to the error state

        ; ****************************
        ; param_err - handle incorrect 
        ; number of data bytes 
        ; ****************************
param_err movlw ill_len ; set up error code
        movwf err_reg
        retlw st_err    ; and exit to the error routine

        ; ****************************
        ; send_err - send '?'<Errcode><CR>
        ; to the serial port
        ; ****************************
send_err movlw 03F    ; Ascii '?'
         call send_rs ; send it
         movf err_reg,W ; get error code
         call sendbyte ; and send it
         goto send_cr  ; and finish with a <CR>
 
        ; ****************************
        ; send_ok - send 'OK'<CR> to 
        ; the serial port
        ; ****************************
send_ok movlw 04F    ; Ascii 'O'
        call send_rs ; send it 
        movlw 04B    ; Ascii 'K'
        call send_rs ; send it
                     ; and fall into send_cr
        
        ; ****************************
        ; send_cr - send Carriage Return
        ; to RS232 port
        ; ****************************
send_cr movlw 0D ; Ascii <CR>
        call send_rs ; send it
        retlw st_cmd ; Exit to the command state (if called from dispatch)

        ; ****************************        
        ; send_rs - Send byte in W to
        ; RS232 output at 9600 baud
        ; ****************************
send_rs bcf porta,TxD ; send start bit
        movwf temp    ; Save character to transmit
        movlw 8       ; 8 bits/char
        movwf bit_cnt
        nop           ; Equalise delay
        call tx_delay ; Wait for one bit time
rstxlp  btfsc temp,0  ; Send LSB of temp to TxD
        bsf porta,TxD
        btfss temp,0
        bcf porta,TxD
        rrf temp,F    ; Shift character right one bit
        call tx_delay ; Wait for 1 bit time
        decfsz bit_cnt,f ; Decrement bit counter
        goto rstxlp      ; Go round again if there are any more bits
        bsf porta,TxD    ; Set TxD to marking state for stop bit
        call tx_delay    ; Wait for stop bit times
        goto tx_delay 

        ; ****************************
        ; rec_rs - Receive 1 byte from
        ; the RS232 port
        ; ****************************
rec_rs  btfss porta,RxD ; Wait for the RxD input to be marking (idle)
        goto rec_rs
rec_st  btfsc porta,RxD ; Now wait for the start bit
        goto rec_st
        call rxhalf     ; OK - there's a start bit. Now move to the 
                        ; centre of it.
        movlw 08        ; 8 bits to receive
        movwf bit_cnt
        clrf temp       ; clear Rx buffer
        bcf status,carry ; clear carry flag (used in rotates)
rec_lp  call rx_delay ; wait 1 bit-time
        rrf temp,F    ; shift character right 1 bit
        btfsc porta,RxD ; test RxD line
        bsf temp,7      ; if spacing, then set MSB of the Rx buffer
        decfsz bit_cnt,F ; decrement bit count
        goto rec_lp ; round again for the next bit
        movf temp,W ; get received byte into W
        return

        ; ****************************
        ; i2c_st - send I2C start
        ; condition (a high-low 
        ; transition of SDA when
        ; SCL is high)
        ; Enter and exit with port registers
        ; selected
        ; ****************************
i2c_st  bsf status,rp ; select tristate registers
        bsf portb,SDA ; make sure that the data line is high
        call clk_hi   ; and that the clock line is high
        bsf status,rp ; select tristate registers again
        bcf portb,SDA ; now bring the data line low
        call i2cdelay ; wait a bit
        bcf portb,SCL ; bring the clock line low
        call i2cdelay ; and wait again
        bcf status,rp ; select the port registers
        return        ; exit

        ; ****************************
        ; i2c_sp - send an I2C stop
        ; condition (a low to high
        ; transition of the SDA line
        ; with SCL high)
        ; Enter and exit with the port
        ; registers selected
        ; ****************************
i2c_sp  bsf status,rp ; select the tristate registers
        bcf portb,SCL ; set clock
        bcf portb,SDA ; and data lines low
        call clk_hi   ; Set the clock line high
        bsf status,rp ; select the tristate registers
        bsf portb,SDA ; set the data line high
        call i2cdelay ; wait a bit
        bcf status,rp ; select port registers
        return        ; exit

        ; ****************************
        ; send_i2c - send byte (in W)
        ; to the I2C bus. Exit
        ; with W containing 0 if
        ; no acknowledge, 1 if acknowledge
        ; Enter and exit with the port 
        ; registers enabled
        ; ****************************
send_i2c movwf temp   ; save character to transmit
        bsf status, rp ; select tristate registers
        bcf portb,SCL ; set clock line low
        movlw 08      ; 8 bits to send
        movwf bit_cnt ; save it in the bit counter
i2c_sl  bcf portb,SDA ; set the data line low
        btfsc temp,7  ; send next bit
        bsf portb,SDA ; if it's a 1, make SDA float high
        call clk_hi   ; now make SCL float high
        bsf status,rp ; select tristate registers again
        bcf portb,SCL ; set clock low
        call i2cdelay ; wait a bit
        rlf temp,F    ; shift the bits along
        decfsz bit_cnt,F ; decrement bit counter
        goto i2c_sl   ; round again for the next bit
        bsf portb,SDA ; let SDA float high for acknowledge time
        call clk_hi   ; set clock high to read the acknowledge bit
        clrf temp     ; clear acknowledge flag
        btfss portb,SDA ; test acknowledge bit
        incf temp,F   ; if there's an acknowledge, set the flag
        bsf status,rp ; select tristate registers
        call i2cdelay ; wait a bit
        bcf portb,SCL ; and set the clock line low again
        bcf status,rp ; select the port registers 
        movf temp,W   ; get acknowledge flag into W
        return        ; and exit

        ; ****************************
        ; rec_i2c - receive 1 byte from 
        ; the I2C port. On entry, W
        ; indicates if an acknowledge
        ; should be given (1 if so).
        ; Exit with character in W
        ; Enter and exit with the
        ; port registers selected
        ; ****************************
rec_i2c movwf temp1    ; save acknowledge flag
        bsf status,rp  ; select tristate registers
        bcf portb,SCL  ; set clock low
        call i2cdelay  ; wait a bit
        bsf portb,SDA  ; and set SDA high
        movlw 08       ; 8 bits to receive
        movwf bit_cnt  ; set up bit counter
        bcf status,carry ; clear carry (used by shifts)
        clrf temp      ; and clear Rx buffer
i2c_rl  call clk_hi    ; set clock line high
        rlf temp,F     ; shift received bits along
        btfsc portb,SDA ; test next data bit
        bsf temp,0     ; set next bit if SDA high
        bsf status,rp  ; select tristate registers
        call i2cdelay  ; wait a bit
        bcf portb,SCL  ; and set clock low
        decfsz bit_cnt,F ; decrement bit counter
        goto i2c_rl    ; and round again for the next bit
        call i2cdelay  ; wait a bit
        movf temp1,F   ; test acknowledge flag
        btfss status,zero ; if acknowledge needed
        bcf portb,SDA  ; bring SDA line low
        call clk_hi    ; make clock high
        bsf status,rp  ; select tristate registers
        bcf portb,SCL  ; and make clock low again
        call i2cdelay  ; wait a bit
        bcf status,rp  ; select port registers
        movf temp,W    ; get received character
        return         ; and exit

        ; ****************************
        ; clk_hi - set I2C clock pin
        ; high
        ; ****************************
clk_hi  call i2cdelay  ; wait a bit
        bsf status,rp  ; select tristate registers
        bsf portb,SCL  ; set clock line to input - pulled high
        bcf status,rp  ; select ports
tst_clk btfss portb,SCL ; check that the clock line is high
        goto tst_clk   ; round again until it is 
        call i2cdelay  ; wait a bit more
        return

        ; ****************************
        ; Delay Routines
        ; ****************************
rxhalf  movlw rxdlyhlf ; Rx half bit delay
        goto delay
rx_delay nop  
        movlw rxdlycnt ; Rx inter-bit delay
        goto delay
i2cdelay movlw i2ccnt ; I2C 
        goto delay 
tx_delay nop
         movlw txdlycnt ; Tx delay counter        
                        ; and fall into delay
delay   movwf dly_cnt   ; Set up delay counter
delaylp decfsz dly_cnt,f ; Decrement delay counter       
        goto delaylp     ; round again until it's 0
        return           ; exit
        end


